<?php
/**
 * Admin: The Dashboard class
 *
 * @link        https://oxyprops.com
 *
 * @package     OxyProps
 * @subpackage  Admin
 * @author      Cédric Bontems <cedric@thewebforge.dev>
 * @since       1.4.0
 * @copyright   Copyright (c) 2022, Cédric Bontems
 * @license     https://www.gnu.org/licenses/gpl-2.0.html  GPLv2 or later
 */

namespace OxyProps\Inc;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * OxyProps\Inc\Dashboard.
 *
 * Creates and handles the plugin dashboard.
 *
 * @since 1.4.0 Creation.
 * @since 1.5.0 Include Singleton Trait and get rid of base controller.
 * @author Cédric Bontems <cedric@thewebforge.dev>
 */
class Dashboard {

	use Singleton;

	/**
	 * Initialize Dashboard class.
	 *
	 * Called by the constructor. It retreives and populates all the class properties.
	 *
	 * @see https://developer.wordpress.org/reference/hooks/admin_menu/
	 * @see https://developer.wordpress.org/reference/hooks/admin_body_class/
	 * @see oxyprops_do_dashboard()
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	public function register(): void {
		// * Actions.
		add_action( 'admin_menu', array( $this, 'oxyprops_register_dashboard' ) );

		// * Filters.
		add_filter( 'admin_body_class', array( $this, 'oxyprops_apply_admin_body_classes' ) );

		$helpers = Init::get_instance( 'helpers' );
		$helpers->brx_mute();
		$helpers->oxy_mute();
	}

	/**
	 * Creates the admin menu page.
	 *
	 * Registers admin menus and pages.
	 *
	 * @see https://developer.wordpress.org/reference/hooks/admin_print_styles-hook_suffix/
	 * @see https://developer.wordpress.org/reference/hooks/admin_print_scripts-hook_suffix/
	 *
	 * @since 1.4.0 Creation.
	 * @since 1.5.0 Fix a case where $core->oxyprops_license_status could be null.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	public function oxyprops_register_dashboard(): void {

		$capability = Helpers::get_user_capability();
		$core       = Init::get_instance( 'core' );
		$main_menu  = Init::get_instance( 'builder' )->get_adapted_name();
		$slug       = Init::get_instance( 'oxyprops' )->get_slug();
		$icon       = Init::get_instance( 'oxyprops' )->get_icon();

		$submenus  = array(
			''              => esc_html__( 'Dashboard', 'oxyprops' ),
			'settings'      => esc_html__( 'Settings', 'oxyprops' ),
			'colors'        => esc_html__( 'Colors', 'oxyprops' ),
			'typography'    => esc_html__( 'Typography', 'oxyprops' ),
			'preview'       => esc_html__( 'Preview', 'oxyprops' ),
			'documentation' => esc_html__( 'Documentation', 'oxyprops' ),
		);
		$dashboard = add_menu_page(
			$main_menu,
			$main_menu,
			$capability,
			$slug,
			array( $this, 'oxyprops_do_dashboard' ),
			$icon,
		);
		if ( $core && \property_exists( $core, 'oxyprops_license_status' ) && $core->oxyprops_license_status && $core->oxyprops_license_status->is_valid ) {
			$submenu_position = 1;
			foreach ( $submenus as $submenu_slug => $submenu_name ) {
				// Generate HashRouter compliant urls.
				$submenu_url = admin_url( 'admin.php?page=oxyprops#/' . $submenu_slug );

				\add_submenu_page(
					$slug,
					$submenu_name,
					$submenu_name,
					$capability,
					$submenu_url,
					null,
					$submenu_position
				);
				$submenu_position++;
			}

			// Remove duplicate first submenu item generated by WP.
			remove_submenu_page( $slug, $slug );
		}
		add_action( "admin_print_styles-$dashboard", array( $this, 'oxyprops_enqueue_dashboard_styles' ) );
		add_action( "admin_print_scripts-$dashboard", array( $this, 'oxyprops_enqueue_dashboard_scripts' ) );
	}

	/**
	 * Adds Dashboard page specific styles.
	 *
	 * @since 1.4.0 Describe changes.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	public function oxyprops_enqueue_dashboard_styles(): void {

		$helpers = Init::get_instance( 'helpers' );
		$path    = Init::get_instance( 'oxyprops' )->get_path();
		$url     = Init::get_instance( 'oxyprops' )->get_url();

		$script_args = include $path . 'build/oxyprops-props.asset.php';
		wp_enqueue_style( 'oxyprops-props', $url . 'build/oxyprops-props.css', array( 'wp-components' ), $script_args['version'], 'all' );

		// Inline settings.
		$user_styles = $helpers->get_oxy_settings() . $helpers->get_user_color_choices() . $helpers->get_user_theme_choices() . $helpers->get_user_typography_choices();
		wp_add_inline_style( 'oxyprops-props', $user_styles );

		// Enqueue the App Built stylesheet.
		$script_args = include $path . 'build/admin-app.asset.php';
		wp_enqueue_style( 'oxyprops-admin-app', $url . 'build/admin-app.css', array( 'wp-components' ), $script_args['version'], 'all' );
	}

	/**
	 * Add admin body classes when needed.
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $classes The original classes list.
	 * @return string The modified classes list.
	 */
	public function oxyprops_apply_admin_body_classes( string $classes ) {
		$dashboard_pages = $this->oxyprops_get_dashboard_pages();
		$current_screen  = get_current_screen();

		if ( in_array( $current_screen->id, $dashboard_pages, true ) ) {
			$classes .= ' oxyprops-dashboard-page';
		}

		return $classes;
	}

	/**
	 * Get a list of our Dashboard pages.
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return array The admin pages.
	 */
	public function oxyprops_get_dashboard_pages(): array {
		return apply_filters(
			'oxyprops_dashboard_screens',
			array(
				'toplevel_page_oxyprops',
			)
		);
	}

	/**
	 * Enqueues dashboard specific scripts.
	 *
	 * @since 1.4.0 Creation.
	 * @since 1.5.0 Use build assets to enqueue scripts.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	public function oxyprops_enqueue_dashboard_scripts(): void {

		$defaults          = Init::get_instance( 'defaults' );
		$url               = Init::get_instance( 'oxyprops' )->get_url();
		$path              = Init::get_instance( 'oxyprops' )->get_path();
		$builder           = Init::get_instance( 'builder' )->get_name();
		$adapted_name      = Init::get_instance( 'builder' )->get_adapted_name();
		$supported_builder = Init::get_instance( 'builder' )->get_supported();

		$script_args = include $path . 'build/admin-app.asset.php';
		wp_enqueue_script(
			'oxyprops-dashboard-app',
			$url . 'build/admin-app.js',
			$script_args['dependencies'],
			$script_args['version'],
			true
		);

		if ( function_exists( 'wp_set_script_translations' ) ) {
			wp_set_script_translations(
				'oxyprops-dashboard-app',
				'oxyprops',
				$path . 'languages/'
			);
		}

		// TODO: Check if Nonce is necessary
		// Add Nonce Midleware for apiFetch.
		wp_add_inline_script(
			'oxyprops-dashboard-app',
			'const apiFetch = wp.apiFetch; apiFetch.use( apiFetch.createNonceMiddleware( "' . wp_create_nonce( 'oxyprops-license-nonce' ) . '" ) )',
		);

		$current_settings = isset( get_option( 'oxyprops', array() )['settings'] ) ? get_option( 'oxyprops', array() )['settings'] : array();

		wp_localize_script(
			'oxyprops-dashboard-app',
			'oxypropsSettings',
			array(
				'settings'          => wp_parse_args(
					$current_settings,
					$defaults->get_oxyprops_default_settings()
				),
				'defaults'          => $defaults->get_oxyprops_default_settings(),
				'builder_name'      => $builder,
				'adapted_name'      => $adapted_name,
				'supported_builder' => $supported_builder,
				'nonce'             => wp_create_nonce( 'oxyprops-app-nonce' ),
				'pluginsUrl'        => admin_url( 'plugins.php' ),
			)
		);
	}

	/**
	 * Adds translation support for javascript scripts.
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	public function translate_scripts() {
		$path = Init::get_instance( 'oxyprops' )->get_path();
		if ( function_exists( 'wp_set_script_translations' ) ) {
			wp_set_script_translations( 'oxyprops-dashboard-app', 'oxyprops', $path . 'languages/' );
		}
	}

	/**
	 * Renders the dashboard markup.
	 *
	 * @since 1.4.0 Creation.
	 * @since 1.5.0 Structure simplification. App rendered directly in this div.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	public function oxyprops_do_dashboard() {
		?>
		<div id="oxyprops-admin-app-container" class="wrap oxyprops-dashboard-wrap"></div>
		<?php
	}
}
